import { Resend } from "resend";

// Initialize Resend client
const resend = new Resend(process.env.RESEND_API_KEY);

// Email configuration
// For testing: Use Resend's test domain (onboarding@resend.dev) - no verification needed
// For production: Use your verified domain (e.g., noreply@yourdomain.com)
const EMAIL_FROM = process.env.EMAIL_FROM || "onboarding@resend.dev";
const EMAIL_FROM_NAME = process.env.EMAIL_FROM_NAME || "Hospital Task System";
function getAppUrl(): string {
  if (process.env.APP_URL) {
    return process.env.APP_URL;
  }

  if (process.env.NEXT_PUBLIC_APP_URL) {
    return process.env.NEXT_PUBLIC_APP_URL;
  }

  if (process.env.VERCEL_URL) {
    return `https://${process.env.VERCEL_URL}`;
  }

  if (process.env.VERCEL_BRANCH_URL) {
    return `https://${process.env.VERCEL_BRANCH_URL}`;
  }

  return "http://localhost:3000";
}

const APP_URL = getAppUrl();

/**
 * Send task assignment notification email
 */
export async function sendTaskAssignmentEmail(
  to: string,
  task: {
    id: string;
    title: string;
    description: string;
    dueDate: Date | null;
    assignedBy?: { name: string; email: string };
  }
) {
  if (!process.env.RESEND_API_KEY) {
    console.warn("RESEND_API_KEY not set, skipping email");
    return;
  }

  try {
    const dueDateText = task.dueDate
      ? new Date(task.dueDate).toLocaleDateString()
      : "No due date";

    const assignedByText = task.assignedBy
      ? `Assigned by: ${task.assignedBy.name} (${task.assignedBy.email})`
      : "";

    const taskUrl = `${APP_URL}/tasks/${task.id}`;

    const { data, error } = await resend.emails.send({
      from: `${EMAIL_FROM_NAME} <${EMAIL_FROM}>`,
      to: [to],
      subject: `New Task Assigned: ${task.title}`,
      html: `
        <!DOCTYPE html>
        <html>
          <head>
            <meta charset="utf-8">
            <style>
              body { font-family: Arial, sans-serif; line-height: 1.6; color: #1e293b; }
              .container { max-width: 600px; margin: 0 auto; padding: 20px; }
              .header { background-color: #008080; color: white; padding: 20px; border-radius: 8px 8px 0 0; }
              .content { background-color: #f9fafb; padding: 20px; border-radius: 0 0 8px 8px; }
              .task-card { background-color: white; padding: 15px; margin: 15px 0; border-left: 4px solid #008080; border-radius: 4px; }
              .button { display: inline-block; padding: 12px 24px; background-color: #008080; color: white; text-decoration: none; border-radius: 6px; margin-top: 15px; }
              .button:hover { background-color: #006666; }
              .footer { margin-top: 20px; padding-top: 20px; border-top: 1px solid #e2e8f0; font-size: 12px; color: #64748b; }
            </style>
          </head>
          <body>
            <div class="container">
              <div class="header">
                <h1>New Task Assigned</h1>
              </div>
              <div class="content">
                <p>Hello,</p>
                <p>You have been assigned a new task:</p>
                <div class="task-card">
                  <h2 style="margin-top: 0;">${task.title}</h2>
                  <p><strong>Description:</strong></p>
                  <p>${task.description}</p>
                  <p><strong>Due Date:</strong> ${dueDateText}</p>
                  ${
                    assignedByText
                      ? `<p><strong>${assignedByText}</strong></p>`
                      : ""
                  }
                </div>
                <a href="${taskUrl}" class="button" style="color: white !important;">View Task</a>
                <div class="footer">
                  <p>This is an automated email from ${EMAIL_FROM_NAME}.</p>
                  <p>If you have any questions, please contact your administrator.</p>
                </div>
              </div>
            </div>
          </body>
        </html>
      `,
    });

    if (error) {
      console.error("Error sending task assignment email:", error);
      throw error;
    }

    return data;
  } catch (error) {
    console.error("Failed to send task assignment email:", error);
    throw error;
  }
}

/**
 * Send ticket assignment notification email
 */
export async function sendTicketAssignmentEmail(
  to: string,
  ticket: {
    id: string;
    title: string;
    description: string;
    severity: number;
    department: string;
    location: string;
    assignedBy?: { name: string; email: string };
  }
) {
  if (!process.env.RESEND_API_KEY) {
    console.warn("RESEND_API_KEY not set, skipping email");
    return;
  }

  try {
    const severityLabels = ["Low", "Medium", "High", "Critical"];
    const severityText = severityLabels[ticket.severity] || "Unknown";

    const assignedByText = ticket.assignedBy
      ? `Assigned by: ${ticket.assignedBy.name} (${ticket.assignedBy.email})`
      : "";

    const ticketUrl = `${APP_URL}/tickets/${ticket.id}`;

    const { data, error } = await resend.emails.send({
      from: `${EMAIL_FROM_NAME} <${EMAIL_FROM}>`,
      to: [to],
      subject: `New Ticket Assigned: ${ticket.title}`,
      html: `
        <!DOCTYPE html>
        <html>
          <head>
            <meta charset="utf-8">
            <style>
              body { font-family: Arial, sans-serif; line-height: 1.6; color: #1e293b; }
              .container { max-width: 600px; margin: 0 auto; padding: 20px; }
              .header { background-color: #008080; color: white; padding: 20px; border-radius: 8px 8px 0 0; }
              .content { background-color: #f9fafb; padding: 20px; border-radius: 0 0 8px 8px; }
              .ticket-card { background-color: white; padding: 15px; margin: 15px 0; border-left: 4px solid #008080; border-radius: 4px; }
              .button { display: inline-block; padding: 12px 24px; background-color: #008080; color: white; text-decoration: none; border-radius: 6px; margin-top: 15px; }
              .button:hover { background-color: #006666; }
              .footer { margin-top: 20px; padding-top: 20px; border-top: 1px solid #e2e8f0; font-size: 12px; color: #64748b; }
            </style>
          </head>
          <body>
            <div class="container">
              <div class="header">
                <h1>New Ticket Assigned</h1>
              </div>
              <div class="content">
                <p>Hello,</p>
                <p>You have been assigned a new ticket:</p>
                <div class="ticket-card">
                  <h2 style="margin-top: 0;">${ticket.title}</h2>
                  <p><strong>Description:</strong></p>
                  <p>${ticket.description}</p>
                  <p><strong>Severity:</strong> ${severityText}</p>
                  <p><strong>Department:</strong> ${ticket.department}</p>
                  <p><strong>Location:</strong> ${ticket.location}</p>
                  ${
                    assignedByText
                      ? `<p><strong>${assignedByText}</strong></p>`
                      : ""
                  }
                </div>
                <a href="${ticketUrl}" class="button" style="color: white !important;">View Ticket</a>
                <div class="footer">
                  <p>This is an automated email from ${EMAIL_FROM_NAME}.</p>
                  <p>If you have any questions, please contact your administrator.</p>
                </div>
              </div>
            </div>
          </body>
        </html>
      `,
    });

    if (error) {
      console.error("Error sending ticket assignment email:", error);
      throw error;
    }

    return data;
  } catch (error) {
    console.error("Failed to send ticket assignment email:", error);
    throw error;
  }
}

/**
 * Send daily task reminder email
 */
export async function sendTaskReminderEmail(
  to: string,
  userName: string,
  tasks: Array<{
    id: string;
    title: string;
    description: string;
    dueDate: Date | null;
    isOverdue: boolean;
    isRecurring: boolean;
    recurrencePattern: string | null;
  }>
) {
  if (!process.env.RESEND_API_KEY) {
    console.warn("RESEND_API_KEY not set, skipping email");
    return;
  }

  if (tasks.length === 0) {
    return; // Don't send email if no tasks
  }

  try {
    const overdueTasks = tasks.filter((t) => t.isOverdue);
    const upcomingTasks = tasks.filter((t) => !t.isOverdue);

    const taskUrl = `${APP_URL}/tasks`;

    const { data, error } = await resend.emails.send({
      from: `${EMAIL_FROM_NAME} <${EMAIL_FROM}>`,
      to: [to],
      subject: `Daily Task Reminder - ${tasks.length} ${
        tasks.length === 1 ? "Task" : "Tasks"
      } Require Attention`,
      html: `
        <!DOCTYPE html>
        <html>
          <head>
            <meta charset="utf-8">
            <style>
              body { font-family: Arial, sans-serif; line-height: 1.6; color: #1e293b; }
              .container { max-width: 600px; margin: 0 auto; padding: 20px; }
              .header { background-color: #008080; color: white; padding: 20px; border-radius: 8px 8px 0 0; }
              .content { background-color: #f9fafb; padding: 20px; border-radius: 0 0 8px 8px; }
              .task-card { background-color: white; padding: 15px; margin: 15px 0; border-left: 4px solid #008080; border-radius: 4px; }
              .overdue-task { border-left-color: #dc2626; }
              .button { display: inline-block; padding: 12px 24px; background-color: #008080; color: white; text-decoration: none; border-radius: 6px; margin-top: 15px; }
              .button:hover { background-color: #006666; }
              .footer { margin-top: 20px; padding-top: 20px; border-top: 1px solid #e2e8f0; font-size: 12px; color: #64748b; }
              .badge { display: inline-block; padding: 4px 8px; border-radius: 4px; font-size: 12px; font-weight: bold; margin-left: 8px; }
              .badge-overdue { background-color: #dc2626; color: white; }
              .badge-recurring { background-color: #14b8a6; color: white; }
            </style>
          </head>
          <body>
            <div class="container">
              <div class="header">
                <h1>Daily Task Reminder</h1>
              </div>
              <div class="content">
                <p>Hello ${userName},</p>
                <p>You have <strong>${tasks.length}</strong> ${
        tasks.length === 1 ? "task" : "tasks"
      } that require your attention:</p>
                ${
                  overdueTasks.length > 0
                    ? `
                  <h3 style="color: #dc2626; margin-top: 20px;">⚠️ Overdue Tasks (${
                    overdueTasks.length
                  })</h3>
                  ${overdueTasks
                    .map(
                      (task) => `
                    <div class="task-card overdue-task">
                      <h3 style="margin-top: 0;">
                        ${task.title}
                        <span class="badge badge-overdue">OVERDUE</span>
                        ${
                          task.isRecurring
                            ? `<span class="badge badge-recurring">${
                                task.recurrencePattern?.toUpperCase() ||
                                "RECURRING"
                              }</span>`
                            : ""
                        }
                      </h3>
                      <p>${task.description}</p>
                      <p><strong>Due Date:</strong> ${
                        task.dueDate
                          ? new Date(task.dueDate).toLocaleDateString()
                          : "No due date"
                      }</p>
                      <a href="${taskUrl}/${
                        task.id
                      }" style="color: #008080; text-decoration: none;">View Task →</a>
                    </div>
                  `
                    )
                    .join("")}
                `
                    : ""
                }
                ${
                  upcomingTasks.length > 0
                    ? `
                  <h3 style="margin-top: 20px;">📋 Incomplete Tasks (${
                    upcomingTasks.length
                  })</h3>
                  ${upcomingTasks
                    .map(
                      (task) => `
                    <div class="task-card">
                      <h3 style="margin-top: 0;">
                        ${task.title}
                        ${
                          task.isRecurring
                            ? `<span class="badge badge-recurring">${
                                task.recurrencePattern?.toUpperCase() ||
                                "RECURRING"
                              }</span>`
                            : ""
                        }
                      </h3>
                      <p>${task.description}</p>
                      <p><strong>Due Date:</strong> ${
                        task.dueDate
                          ? new Date(task.dueDate).toLocaleDateString()
                          : "No due date"
                      }</p>
                      <a href="${taskUrl}/${
                        task.id
                      }" style="color: #008080; text-decoration: none;">View Task →</a>
                    </div>
                  `
                    )
                    .join("")}
                `
                    : ""
                }
                <a href="${taskUrl}" class="button" style="color: white !important;">View All Tasks</a>
                <div class="footer">
                  <p>This is an automated daily reminder from ${EMAIL_FROM_NAME}.</p>
                  <p>You will receive this email every weekday morning at 9:00 AM for incomplete and overdue tasks.</p>
                </div>
              </div>
            </div>
          </body>
        </html>
      `,
    });

    if (error) {
      console.error("Error sending task reminder email:", error);
      throw error;
    }

    return data;
  } catch (error) {
    console.error("Failed to send task reminder email:", error);
    throw error;
  }
}

/**
 * Send problem report email to task creator
 */
export async function sendProblemReportEmail(
  to: string,
  task: {
    id: string;
    title: string;
    description: string;
    createdBy?: { name: string | null; email: string };
  },
  element: {
    id: string;
    name: string;
  },
  reportedBy: {
    name: string | null;
    email: string;
  },
  notes?: string | null,
  ticketId?: string | null
) {
  if (!process.env.RESEND_API_KEY) {
    console.warn("RESEND_API_KEY not set, skipping email");
    return;
  }

  try {
    const taskUrl = `${APP_URL}/tasks/${task.id}`;
    const reportedByName = reportedBy.name || reportedBy.email || "Unknown";
    const ticketInfo = ticketId
      ? `<p><strong>Related Ticket ID:</strong> ${ticketId}</p>`
      : "";
    const isGeneralTask = element.id === task.id;
    const problemIntro = isGeneralTask
      ? "A problem has been reported with your task:"
      : "A problem has been reported for an element in your task:";
    const problemLabel = isGeneralTask
      ? "Problem Reported:"
      : "Element with Problem:";

    const { data, error } = await resend.emails.send({
      from: `${EMAIL_FROM_NAME} <${EMAIL_FROM}>`,
      to: [to],
      subject: isGeneralTask
        ? `Problem Reported: ${task.title}`
        : `Problem Reported: ${element.name} - ${task.title}`,
      html: `
        <!DOCTYPE html>
        <html>
          <head>
            <meta charset="utf-8">
            <style>
              body { font-family: Arial, sans-serif; line-height: 1.6; color: #1e293b; }
              .container { max-width: 600px; margin: 0 auto; padding: 20px; }
              .header { background-color: #dc2626; color: white; padding: 20px; border-radius: 8px 8px 0 0; }
              .content { background-color: #f9fafb; padding: 20px; border-radius: 0 0 8px 8px; }
              .problem-card { background-color: white; padding: 15px; margin: 15px 0; border-left: 4px solid #dc2626; border-radius: 4px; }
              .button { display: inline-block; padding: 12px 24px; background-color: #dc2626; color: white; text-decoration: none; border-radius: 6px; margin-top: 15px; }
              .button:hover { background-color: #b91c1c; }
              .footer { margin-top: 20px; padding-top: 20px; border-top: 1px solid #e2e8f0; font-size: 12px; color: #64748b; }
              .badge { display: inline-block; padding: 4px 8px; border-radius: 4px; font-size: 12px; font-weight: bold; background-color: #dc2626; color: white; }
            </style>
          </head>
          <body>
            <div class="container">
              <div class="header">
                <h1>⚠️ Problem Reported</h1>
              </div>
              <div class="content">
                <p>Hello,</p>
                <p>${problemIntro}</p>
                <div class="problem-card">
                  <h2 style="margin-top: 0;">${task.title}</h2>
                  <p><strong>Task Description:</strong></p>
                  <p>${task.description}</p>
                  ${
                    !isGeneralTask
                      ? `<hr style="margin: 15px 0; border: none; border-top: 1px solid #e2e8f0;">`
                      : ""
                  }
                  ${
                    !isGeneralTask
                      ? `<p><strong>${problemLabel}</strong></p><p><span class="badge">NOT OK</span> <strong>${element.name}</strong></p>`
                      : `<p><span class="badge">NOT OK</span> <strong>Problem Reported</strong></p>`
                  }
                  ${
                    notes
                      ? `<p><strong>Problem Details:</strong></p><p>${notes}</p>`
                      : ""
                  }
                  ${ticketInfo}
                  <p><strong>Reported by:</strong> ${reportedByName} (${
        reportedBy.email
      })</p>
                  <p><strong>Reported at:</strong> ${new Date().toLocaleString()}</p>
                </div>
                <a href="${taskUrl}" class="button" style="color: white !important;">View Task</a>
                <div class="footer">
                  <p>This is an automated email from ${EMAIL_FROM_NAME}.</p>
                  <p>If you have any questions, please contact your administrator.</p>
                </div>
              </div>
            </div>
          </body>
        </html>
      `,
    });

    if (error) {
      console.error("Error sending problem report email:", error);
      throw error;
    }

    return data;
  } catch (error) {
    console.error("Failed to send problem report email:", error);
    throw error;
  }
}

/**
 * Send welcome email with password reset link to new technician
 */
export async function sendWelcomeEmail(
  to: string,
  userName: string | null,
  resetToken: string
) {
  if (!process.env.RESEND_API_KEY) {
    console.warn("RESEND_API_KEY not set, skipping email");
    return;
  }

  try {
    const resetUrl = `${APP_URL}/reset-password?token=${resetToken}`;
    const displayName = userName || to;

    const { data, error } = await resend.emails.send({
      from: `${EMAIL_FROM_NAME} <${EMAIL_FROM}>`,
      to: [to],
      subject: `Welcome to ${EMAIL_FROM_NAME} - Set Your Password`,
      html: `
        <!DOCTYPE html>
        <html>
          <head>
            <meta charset="utf-8">
            <style>
              body { font-family: Arial, sans-serif; line-height: 1.6; color: #1e293b; }
              .container { max-width: 600px; margin: 0 auto; padding: 20px; }
              .header { background-color: #008080; color: white; padding: 20px; border-radius: 8px 8px 0 0; }
              .content { background-color: #f9fafb; padding: 20px; border-radius: 0 0 8px 8px; }
              .button { display: inline-block; padding: 12px 24px; background-color: #008080; color: white; text-decoration: none; border-radius: 6px; margin-top: 15px; }
              .button:hover { background-color: #006666; }
              .footer { margin-top: 20px; padding-top: 20px; border-top: 1px solid #e2e8f0; font-size: 12px; color: #64748b; }
              .warning { background-color: #fef3c7; border-left: 4px solid #f59e0b; padding: 12px; margin: 15px 0; border-radius: 4px; }
            </style>
          </head>
          <body>
            <div class="container">
              <div class="header">
                <h1>Welcome to ${EMAIL_FROM_NAME}!</h1>
              </div>
              <div class="content">
                <p>Hello ${displayName},</p>
                <p>Your account has been created. To get started, please set your password by clicking the button below:</p>
                <a href="${resetUrl}" class="button" style="color: white !important;">Set Your Password</a>
                <div class="warning">
                  <p><strong>⚠️ Important:</strong> This link will expire in 48 hours. If you need a new link, please contact your administrator.</p>
                </div>
                <p>If the button doesn't work, you can copy and paste this link into your browser:</p>
                <p style="word-break: break-all; color: #64748b; font-size: 12px;">${resetUrl}</p>
                <div class="footer">
                  <p>This is an automated email from ${EMAIL_FROM_NAME}.</p>
                  <p>If you didn't expect this email, please contact your administrator.</p>
                </div>
              </div>
            </div>
          </body>
        </html>
      `,
    });

    if (error) {
      console.error("Error sending welcome email:", error);
      throw error;
    }

    return data;
  } catch (error) {
    console.error("Failed to send welcome email:", error);
    throw error;
  }
}

/**
 * Send password reset email
 */
export async function sendPasswordResetEmail(
  to: string,
  userName: string | null,
  resetToken: string
) {
  if (!process.env.RESEND_API_KEY) {
    console.warn("RESEND_API_KEY not set, skipping email");
    return;
  }

  try {
    const resetUrl = `${APP_URL}/reset-password?token=${resetToken}`;
    const displayName = userName || to;

    const { data, error } = await resend.emails.send({
      from: `${EMAIL_FROM_NAME} <${EMAIL_FROM}>`,
      to: [to],
      subject: `Reset Your Password - ${EMAIL_FROM_NAME}`,
      html: `
        <!DOCTYPE html>
        <html>
          <head>
            <meta charset="utf-8">
            <style>
              body { font-family: Arial, sans-serif; line-height: 1.6; color: #1e293b; }
              .container { max-width: 600px; margin: 0 auto; padding: 20px; }
              .header { background-color: #008080; color: white; padding: 20px; border-radius: 8px 8px 0 0; }
              .content { background-color: #f9fafb; padding: 20px; border-radius: 0 0 8px 8px; }
              .button { display: inline-block; padding: 12px 24px; background-color: #008080; color: white; text-decoration: none; border-radius: 6px; margin-top: 15px; }
              .button:hover { background-color: #006666; }
              .footer { margin-top: 20px; padding-top: 20px; border-top: 1px solid #e2e8f0; font-size: 12px; color: #64748b; }
              .warning { background-color: #fef3c7; border-left: 4px solid #f59e0b; padding: 12px; margin: 15px 0; border-radius: 4px; }
            </style>
          </head>
          <body>
            <div class="container">
              <div class="header">
                <h1>Password Reset Request</h1>
              </div>
              <div class="content">
                <p>Hello ${displayName},</p>
                <p>We received a request to reset your password. Click the button below to set a new password:</p>
                <a href="${resetUrl}" class="button" style="color: white !important;">Reset Password</a>
                <div class="warning">
                  <p><strong>⚠️ Important:</strong> This link will expire in 1 hour. If you didn't request this, please ignore this email.</p>
                </div>
                <p>If the button doesn't work, you can copy and paste this link into your browser:</p>
                <p style="word-break: break-all; color: #64748b; font-size: 12px;">${resetUrl}</p>
                <div class="footer">
                  <p>This is an automated email from ${EMAIL_FROM_NAME}.</p>
                  <p>If you didn't request a password reset, please contact your administrator.</p>
                </div>
              </div>
            </div>
          </body>
        </html>
      `,
    });

    if (error) {
      console.error("Error sending password reset email:", error);
      throw error;
    }

    return data;
  } catch (error) {
    console.error("Failed to send password reset email:", error);
    throw error;
  }
}
